// -*- mode: js; js-indent-level: 4; indent-tabs-mode: nil -*-

const Lang = imports.lang;
const Cinnamon = imports.gi.Cinnamon;
const St = imports.gi.St;
const GLib = imports.gi.GLib;

const Main = imports.ui.main;
const MessageTray = imports.ui.messageTray;
const AppletManager = imports.ui.appletManager;
const SignalManager = imports.misc.signalManager;
const Mainloop = imports.mainloop;

function WindowAttentionHandler() {
    this._init();
}

WindowAttentionHandler.prototype = {
    _init : function() {
        this._tracker = Cinnamon.WindowTracker.get_default();

        Mainloop.timeout_add_seconds(4, () => {
            log("Enabling WindowAttentionHandler");
            global.display.connect('window-demands-attention', Lang.bind(this, this._onWindowDemandsAttention));
            global.display.connect('window-marked-urgent', Lang.bind(this, this._onWindowDemandsAttention));

            return GLib.SOURCE_REMOVE;
        })
    },

    _onWindowDemandsAttention : function(display, window) {
        /* Ordinarily, new windows that don't specifically demand focus (like ones created
         * without user interaction) only get some indicator that the window wants the
         * user's attention (like blinking in the window list). Some windows aren't
         * directly tied to a user action, but are generated by the user's action anyhow -
         * these we make an exception for and focus them. */
        if (!window || window.has_focus() || window.is_skip_taskbar() || !Main.isInteresting(window)) {
            return;
        }

        let wmclass = window.get_wm_class();

        if (wmclass) {
            let ignored_classes = global.settings.get_strv("demands-attention-passthru-wm-classes");

            for (let i = 0; i < ignored_classes.length; i++) {
                if (wmclass.toLowerCase().includes(ignored_classes[i].toLowerCase())) {
                    window.activate(global.get_current_time());
                    return;
                }
            }
        }
    }
};

var WindowHandlerSource = class extends MessageTray.Source {
    constructor(app, window) {
        super(app.get_name());
        this.window = window;
        this.app = app;

        this._signals = new SignalManager.SignalManager();

        this._signals.connect(this.window, 'notify::demands-attention', () => this.windowStateChanged());
        this._signals.connect(this.window, 'notify::urgent', () => this.windowStateChanged());
        this._signals.connect(this.window, 'focus', () => this.destroy());
        this._signals.connect(this.window, 'unmanaged', () => this.destroy());
    }

    createNotificationIcon() {
        if (this.app !== null) {
            return this.app.create_icon_texture(this.ICON_SIZE);
        }

        return new St.Icon({ icon_name: 'dialog-information',
                             icon_type: St.IconType.SYMBOLIC,
                             icon_size: this.ICON_SIZE });
    }

    windowStateChanged() {
        if (!this.window.demands_attention && !this.window.urgent) {
            this.destroy();
        }
    }

    open() {
        this.window.activate(global.get_current_time());
    }

    destroy(params) {
        this._signals.disconnectAllSignals();
    }
};


