"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const fs_1 = __importDefault(require("fs"));
const js_yaml_1 = __importDefault(require("js-yaml"));
const mustache_1 = __importDefault(require("mustache"));
const fsPromises = __importStar(require("node:fs/promises"));
const path = __importStar(require("node:path"));
const converter_1 = __importDefault(require("../classes/converter"));
const utils_1 = __importDefault(require("../classes/utils"));
const utils_2 = require("../lib/utils");
/**
 * Comando per creare pacchetti Debian
 */
class Deb extends core_1.Command {
    here = '';
    pathSource = '';
    static args = {
        pathSource: core_1.Args.string({ description: 'pathSource', name: 'pathSource', required: false }),
    };
    static description = 'Create a deb package from your npm package';
    static flags = {
        help: core_1.Flags.help({ char: 'h' }),
        all: core_1.Flags.boolean({ char: 'a', description: 'all architectures' }),
        release: core_1.Flags.string({ char: 'r', description: 'release' }),
        manpages: core_1.Flags.boolean({ char: 'M', description: 'refresh manpages on the sources' }),
        verbose: core_1.Flags.boolean({ char: 'v', description: 'verbose' }),
    };
    static summary = 'Pack CLI into debian package.';
    /**
     * Metodo principale eseguito dal comando
     */
    async run() {
        const { args, flags } = await this.parse(Deb);
        if (process.platform !== 'linux') {
            this.log('debian packing must be run on linux');
            this.exit(0);
        }
        const { verbose, all, manpages } = flags;
        const release = flags.release ?? '1';
        const echo = utils_1.default.setEcho(verbose);
        // Imposta i percorsi di base
        this.here = process.cwd();
        this.pathSource = args.pathSource ? path.resolve(args.pathSource) : this.here;
        // Gestione primo avvio e creazione configurazione perrisbrewery
        if (!fs_1.default.existsSync(path.join(this.here, 'perrisbrewery'))) {
            this.log('perrisbrewery configurations not found, creating sample...');
            fs_1.default.mkdirSync(path.join(this.here, 'perrisbrewery'));
            await (0, utils_2.exec)(`cp -r ${path.resolve(__dirname, '../perrisbrewery.sample/*')} ${path.join(this.here, 'perrisbrewery')}`, echo);
            this.log('perrisbrewery dir created. Edit configuration, include /perrisbrewery/workdir in .gitignore, then run again.');
            this.exit(0);
        }
        if (!fs_1.default.existsSync(path.join(this.pathSource, 'package.json'))) {
            this.log(`package.json not found in ${this.pathSource}`);
            this.exit(0);
        }
        // --- LOGICA DELLE DIRECTORY DI BUILD ---
        const tempBuildRoot = path.join(this.here, 'build'); // Area di assemblaggio temporanea
        const finalReleasesDir = path.join(this.here, 'releases'); // Area per i pacchetti .deb finali
        // Pulisci SOLO la directory di assemblaggio temporanea prima di iniziare
        this.log(`Cleaning temporary build directory: ${tempBuildRoot}`);
        if (fs_1.default.existsSync(tempBuildRoot)) {
            await fsPromises.rm(tempBuildRoot, { recursive: true, force: true });
        }
        await fsPromises.mkdir(tempBuildRoot, { recursive: true });
        // Assicurati che la directory per i pacchetti finali esista
        if (!fs_1.default.existsSync(finalReleasesDir)) {
            await fsPromises.mkdir(finalReleasesDir, { recursive: true });
        }
        // --- FINE LOGICA DIRECTORY ---
        // Determina le architetture da costruire
        let debArchs = [process.arch === 'x64' ? 'amd64' : process.arch];
        if (all) {
            debArchs = ['amd64', 'arm64', 'i386'];
        }
        // Avvia il ciclo di build per ogni architettura
        for (const debArch of debArchs) {
            await this.createPackage(debArch, release, manpages, verbose, tempBuildRoot, finalReleasesDir);
        }
        this.log('All builds complete!');
    }
    /**
     * Crea un singolo pacchetto per una specifica architettura
     */
    async createPackage(debArch, release, manpages, verbose, tempBuildRoot, finalReleasesDir) {
        const echo = utils_1.default.setEcho(verbose);
        this.log('');
        this.log(`--- Building for architecture: ${debArch} ---`);
        const packageJsonContent = fs_1.default.readFileSync(path.join(this.pathSource, 'package.json'), 'utf8');
        const packageJson = JSON.parse(packageJsonContent);
        const { description, version: packageVersion, files } = packageJson;
        const packageName = packageJson.name;
        const mantainer = packageJson.mantainer ?? packageJson.author ?? "Perris' Brewery";
        const binName = Object.keys(packageJson.bin)[0];
        const packageNameVersioned = `${packageName}_${packageVersion}-${release}_${debArch}`;
        // Il pacchetto viene assemblato dentro la directory `build`
        const packageDir = path.join(tempBuildRoot, packageNameVersioned);
        const rootLib = path.join(packageDir, 'usr', 'lib', packageName);
        // Crea la struttura di base del pacchetto
        await Promise.all([
            fsPromises.mkdir(path.join(packageDir, 'DEBIAN'), { recursive: true }),
            fsPromises.mkdir(path.join(packageDir, 'usr', 'bin'), { recursive: true }),
            fsPromises.mkdir(rootLib, { recursive: true }),
            fsPromises.mkdir(path.join(rootLib, 'manpages', 'doc', 'man'), { recursive: true }),
        ]);
        this.log('Package skeleton created.');
        // Gestione dipendenze
        const dependenciesFile = path.join(this.here, 'perrisbrewery', 'template', 'dependencies.yaml');
        const fileContents = fs_1.default.readFileSync(dependenciesFile, 'utf8');
        const dep = js_yaml_1.default.load(fileContents);
        const packages = [...dep.common, ...(dep.arch[debArch] || [])].sort();
        // Crea il file DEBIAN/control
        const template = fs_1.default.readFileSync(path.join(this.here, 'perrisbrewery', 'template', 'control.template'), 'utf8');
        fs_1.default.writeFileSync(path.join(packageDir, 'DEBIAN', 'control'), mustache_1.default.render(template, {
            arch: debArch,
            depends: packages.join(', '),
            description,
            mantainer,
            name: packageName,
            version: `${packageVersion}-${release}`,
        }));
        this.log('DEBIAN/control file created.');
        // Copia script di post-installazione, etc.
        await (0, utils_2.exec)(`cp ${path.join(this.here, 'perrisbrewery', 'scripts', '*')} ${path.join(packageDir, 'DEBIAN/')}`, echo);
        // Crea man page
        const converter = new converter_1.default(path.join(this.pathSource, 'README.md'));
        await converter.readme2md(packageDir, packageName, packageVersion, binName, packageNameVersioned, verbose);
        await converter.md2man(packageDir, packageName, packageVersion, binName, verbose);
        await converter.md2html(packageDir, packageName, packageVersion, binName, verbose);
        this.log('Man page created.');
        // Aggiorna le man page nei sorgenti se richiesto
        if (manpages) {
            this.log('Refreshing manpages in source directory...');
            await (0, utils_2.exec)(`rm -rf ${path.join(this.here, 'manpages')}`, echo);
            await (0, utils_2.exec)(`cp -r ${path.join(rootLib, 'manpages')} ${this.here}`, echo);
        }
        // Copia i file del pacchetto (da `dist`, `node_modules`, ecc.)
        this.log('Copying application files...');
        for (const file of files) {
            await (0, utils_2.exec)(`cp -r ${path.join(this.pathSource, file)} ${rootLib}/`, echo);
        }
        await (0, utils_2.exec)(`cp -r ${path.join(this.pathSource, 'LICENSE')} ${rootLib}`, echo);
        await (0, utils_2.exec)(`cp -r ${path.join(this.pathSource, 'node_modules')} ${rootLib}`, echo);
        await (0, utils_2.exec)(`cp -r ${path.join(this.pathSource, 'package.json')} ${rootLib}`, echo);
        // Copia lock files se esistono
        for (const lockFile of ['pnpm-lock.yaml', 'yarn.lock', 'npm-shrinkwrap.json']) {
            if (fs_1.default.existsSync(path.join(this.pathSource, lockFile))) {
                await (0, utils_2.exec)(`cp -r ${path.join(this.pathSource, lockFile)} ${rootLib}`, echo);
            }
        }
        // Crea il link simbolico per l'eseguibile
        await (0, utils_2.exec)(`ln -sf ../lib/${packageName}/bin/run.js ${path.join(packageDir, 'usr', 'bin', binName)}`);
        // Imposta i permessi corretti e builda il pacchetto
        this.log('Setting permissions and building package...');
        await (0, utils_2.exec)(`sudo chown -R root:root "${packageDir}"`);
        // --- CORREZIONE 2: Cambiamo cartella con 'cd' perché 'exec' non supporta l'opzione 'cwd' ---
        await (0, utils_2.exec)(`cd ${tempBuildRoot} && dpkg-deb --build ${path.basename(packageDir)}`);
        await (0, utils_2.exec)(`sudo rm -rf ${packageDir}`); // Pulisce la dir di assemblaggio per questa arch
        this.log(`Package ${packageNameVersioned}.deb built successfully.`);
        // Crea il checksum e sposta i file finali
        this.log(`Creating checksum and moving to ${finalReleasesDir}...`);
        const originalDir = process.cwd();
        process.chdir(tempBuildRoot);
        await (0, utils_2.exec)(`sha256sum ${packageNameVersioned}.deb > ${packageNameVersioned}.deb.sha256`);
        await (0, utils_2.exec)(`mv ${packageNameVersioned}.deb ${finalReleasesDir}/`);
        await (0, utils_2.exec)(`mv ${packageNameVersioned}.deb.sha256 ${finalReleasesDir}/`);
        process.chdir(originalDir);
        this.log(`--- Finished ${debArch} ---`);
    }
}
exports.default = Deb;
